package org.msh.tb.reports2.variables.concat;

import org.msh.reports.filters.Filter;
import org.msh.reports.variables.Variable;
import org.msh.tb.reports2.ReportResources;
import org.msh.utils.Tuple;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Parse variables name, parameters and filters from a string representation to
 * be used in an instance of {@link ConcatVariable}
 *
 * Created by rmemoria on 14/12/16.
 */
public class ConcatVariableParser {

    /**
     * Parse variables, including parameters and filters
     * @param varid the string containing the variable data
     * @return
     */
    public static ParsedData parse(String varid) {
        List<String> declarations = split(varid, ',');

        ParsedData res = new ParsedData();

        List<ConcatVariableInfo> vars = new ArrayList<ConcatVariableInfo>();
        for (String v: declarations) {
            // first check if it is a meta data definition (like group keys)
            if (!checkMetadata(res, v)) {
                ConcatVariableInfo vinfo = parseVariableInfo(v);
                vars.add(vinfo);
            }
        }

        res.setVariables(vars);

        return res;
    }

    /**
     * Parse a single variable definition, including filters and parameters
     * @param s
     * @return
     */
    private static ConcatVariableInfo parseVariableInfo(String s) {
        List<String> params = split(s, ' ');

        ConcatVariableInfo vinfo = new ConcatVariableInfo();

        // get variable
        Variable var = ReportResources.instance().findVariableById(params.get(0));
        if (var == null) {
            throw new RuntimeException("Variable not found: " + params.get(0));
        }
        vinfo.setVariable(var);

        parseParams(vinfo, params);

        return vinfo;
    }

    /**
     * Check if variable is in fact a meta data definition
     * @param data the parsed data to be returned to the caller
     * @param s the variable definition
     * @return
     */
    private static boolean checkMetadata(ParsedData data, String s) {
        if (s.startsWith("groups ")) {
            parseGroupKeys(data, s.substring(6));
            return true;
        }

        return false;
    }

    private static void parseGroupKeys(ParsedData data, String declaration) {
        List<String> tokens = split(declaration, ' ');
        Map<String, String> groups = new HashMap<String, String>();

        for (String s: tokens) {
            List<String> items = split(s, '=');
            if (items.size() != 2) {
                throw new RuntimeException("Invalid group key definition: " + s);
            }
            String key = removeQuotationMarks(items.get(0));
            String title = removeQuotationMarks(items.get(1));
            groups.put(key, title);
        }
        data.setGroups(groups);
    }

    /**
     * Parse the params of a variable definition
     * @param info the instance of {@link ConcatVariableInfo} that will receive the parameters
     * @param params the list of parameters
     */
    private static void parseParams(ConcatVariableInfo info, List<String> params) {
        for (int i = 1; i < params.size(); i++) {
            String param = params.get(i);

            if ("total".equals(param)) {
                info.setCalcTotal(true);
            } else if (param.startsWith("key=")) {
                handleKeyTitles(info, param);
            } else if (param.startsWith("group=")) {
                handleVariableGroup(info, param);
            } else {
                handleFilter(info, param);
            }
        }
    }

    private static void handleVariableGroup(ConcatVariableInfo info, String param) {
        int pos = param.indexOf("=");
        String p = param.substring(pos + 1);
        info.setGroupKey(p.trim());
    }

    private static void handleFilter(ConcatVariableInfo info, String param) {
        Tuple<String, String> t = parseFilter(param);

        // any filter value found ?
        if (t != null) {
            if (info.getFilters() == null) {
                info.setFilters(new HashMap<Filter, String>());
            }

            Filter filter = ReportResources.instance().findFilterById(t.getValue1());
            if (filter == null) {
                throw new RuntimeException("Filter not found: " + t.getValue1());
            }

            // add filter to the list of filters
            info.getFilters().put(filter, t.getValue2());
        }
    }

    private static void handleKeyTitles(ConcatVariableInfo info, String param) {
        int pos = param.indexOf("=");
        String p = param.substring(pos + 1);
        String[] s = p.split(":");
        if (s.length != 2) {
            throw new RuntimeException("Invalid param: " + param);
        }

        if (info.getKeyTitles() == null) {
            info.setKeyTitles(new HashMap<String, String>());
        }

        String title = removeQuotationMarks(s[1]);
        info.getKeyTitles().put(s[0], title);
    }

    private static String removeQuotationMarks(String s) {
        return s.startsWith("'") && s.endsWith("'") ? s.substring(1, s.length() - 1) : s;
    }

    /**
     * Parse filter name and value
     * @param s the filter definition in the format name=value
     * @return {@link Tuple} containing the name and the value
     */
    private static Tuple<String, String> parseFilter(String s) {
        int pos = s.indexOf("=");
        if (pos == -1) {
            return null;
        }

        String fname = s.substring(0, pos);
        String fvalue = s.substring(pos + 1);

        fvalue = removeQuotationMarks(fvalue);

        return Tuple.of(fname, fvalue);
    }

    /**
     * Split the string by the occurrence of the given character preserving
     * values between quotation marks
     * @param varid
     * @param ch
     * @return
     */
    private static List<String> split(String varid, char ch) {
        List<String> vars = new ArrayList<String>();

        int ini = 0;
        int i = 0;
        while (i < varid.length()) {
            char c = varid.charAt(i);
            if (c == ch) {
                if (i > ini) {
                    vars.add(varid.substring(ini, i));
                }
                i++;
                ini = i;
            } else {
                switch (c) {
                    case '\'':
                        int end = varid.indexOf('\'', i + 1);
                        if (end > 0) {
                            i = end + 1;
                        } else {
                            i++;
                        }
                        break;
                    default: i++;
                }
            }
        }

        String id = varid.substring(ini);
        vars.add(id);

        return vars;
    }

}
